using System;
using System.Data;
using System.Reflection;

using gov.va.med.vbecs.Common;

using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.TimeZoneConfig;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// The class represents time zone and daylight savings configuration 
	/// used for VAMC division. 
	/// </summary>

	#endregion

	public class TimeZoneConfig : BaseBusinessObject
	{
		private DaylightSavingsType _daylightSavingsType;
		private TimeZone _timeZone;

		private DateTime _customDaylightSavingsStart;
		private DateTime _customDaylightSavingsEnd;

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4795"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Instantiated object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4796"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructs an empty instance of a class.
		/// </summary>
		public TimeZoneConfig() : base()
		{
			this.SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );

			_daylightSavingsType = DaylightSavingsType.Unknown;
			_timeZone = null;

			_customDaylightSavingsStart = StNullConvert.DateTimeNull;
			_customDaylightSavingsEnd = StNullConvert.DateTimeNull;

			OnPropertyChanged();

			IsNew = true;
			IsDirty = false;
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4805"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>TimeZoneConfig object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4806"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Constructs an instance of the class loading required data from a given <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow">Source <see cref="DataRow"/> to load data from.</param>
		public TimeZoneConfig( DataRow dtRow ) : this()
		{
			this.LoadFromDataRow( dtRow );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4807"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Populated DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4808"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );
			
			RequireColumns( dtRow );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( TimeZoneConfig ).Name ).ResString ) );

			dtRow[ DivTable.TimeZoneId ] = StDbNullConvert.From( _timeZone.ID );
			dtRow[ DivTable.DaylightSavingsTypeId ] = DaylightSavingsTypeDefinition.GetDaylightSavingsTypeId( _daylightSavingsType );
			
			LoadCustomDaylightSavinsConfigToDataRow( dtRow );

			return dtRow;
		}

		private void LoadCustomDaylightSavinsConfigToDataRow( DataRow dtRow )
		{
			if( _daylightSavingsType == DaylightSavingsType.Custom )
			{
				dtRow[ DivTable.DaylightSavingsStartDate ] = StDbNullConvert.From( _customDaylightSavingsStart );
				dtRow[ DivTable.DaylightSavingsEndDate ] = StDbNullConvert.From( _customDaylightSavingsEnd );
			}
			else
			{
				dtRow[ DivTable.DaylightSavingsStartDate ] = StDbNullConvert.From( StNullConvert.DateTimeNull );
				dtRow[ DivTable.DaylightSavingsEndDate ] = StDbNullConvert.From( StNullConvert.DateTimeNull );
			}
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			RequireColumns( dtRow );
			Utility.RequireNonNullColumns( dtRow, DivTable.TimeZoneId, DivTable.DaylightSavingsTypeId );

			_timeZone = new TimeZone( StDbNullConvert.ToChar( dtRow[ DivTable.TimeZoneId ] ) );
			_daylightSavingsType = DaylightSavingsTypeDefinition.GetDaylightSavingsTypeFromId( StDbNullConvert.ToInt32( dtRow[ DivTable.DaylightSavingsTypeId ] ) );

			_customDaylightSavingsStart = StDbNullConvert.ToDateTime( dtRow[ DivTable.DaylightSavingsStartDate ] );
			_customDaylightSavingsEnd = StDbNullConvert.ToDateTime( dtRow[ DivTable.DaylightSavingsEndDate ] );

			OnPropertyChanged();

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.InvalidDataLoadedFromDataRowIntoBusinessObject( typeof( TimeZoneConfig ).Name, MethodBase.GetCurrentMethod().Name ).ResString ) );

			IsNew = false;
			IsDirty = false;
		}



		/// <summary>
		/// Validates supplied <see cref="DataRow"/> ensuring that all required columns are present.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> used for data exchange between layers.</param>
		protected virtual void RequireColumns( DataRow dtRow )
		{
			Utility.RequireColumns( dtRow, DivTable.TimeZoneId, DivTable.DaylightSavingsTypeId, DivTable.DaylightSavingsStartDate, DivTable.DaylightSavingsEndDate  );
		}

		/// <summary>
		/// Runs batch validation on all business rules associated with the class. 
		/// It's run in batch since there're rules dependent on multiple fields, having 
		/// in-accessors validation would create code duplication.
		/// </summary>
		protected virtual void ValidateAllRules()
		{
			RuleBroken( BrRules.TimeZoneNotSet, _timeZone == null );

			bool _checkCustomDst = !RuleBroken( BrRules.DaylightSavingsTypeNotSet, _daylightSavingsType == DaylightSavingsType.Unknown ) &&
				_daylightSavingsType == DaylightSavingsType.Custom;

			bool _customDstDatesUndefined = 
				RuleBroken( BrRules.CustomDaylightSavingsStartNotSet, _checkCustomDst && StNullConvert.IsNull( _customDaylightSavingsStart ) ) || 
				RuleBroken( BrRules.CustomDaylightSavingsEndNotSet, _checkCustomDst && StNullConvert.IsNull( _customDaylightSavingsEnd ) );

			RuleBroken( 
				BrRules.CustomDaylightSavingsStartGreatherThanEnd, 
				_checkCustomDst && !_customDstDatesUndefined && ( _customDaylightSavingsStart.DayOfYear >= _customDaylightSavingsEnd.DayOfYear ) );
		}

		/// <summary>
		/// Should be called whenever any field value is changed via property. 
		/// Marks object dirty and runs validation.
		/// </summary>
		protected virtual void OnPropertyChanged()
		{
			IsDirty = true;

			ValidateAllRules();
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4797"> 
		///		<ExpectedInput>Valid time zone.</ExpectedInput>
		///		<ExpectedOutput>Supplied time zone, corresponding rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4798"> 
		///		<ExpectedInput>Null.</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Time zone. 
		/// </summary>
		public TimeZone TimeZone
		{
			get
			{
				return _timeZone;
			}
			set
			{
				if( _timeZone == value )
					return;

				_timeZone = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4799"> 
		///		<ExpectedInput>Test daylight savings type.</ExpectedInput>
		///		<ExpectedOutput>Assigned daylight savings type, corresponding rule is not broken.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4800"> 
		///		<ExpectedInput>Unknown daylight savings type.</ExpectedInput>
		///		<ExpectedOutput>Corresponding rule is broken.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Division daylight savings type.
		/// </summary>
		public DaylightSavingsType DaylightSavingsType
		{
			get
			{
				return _daylightSavingsType;
			}
			set 
			{
				if( _daylightSavingsType == value )
					return;

				_daylightSavingsType = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4801"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4802"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Start date for daylight savings time for a given year.
		/// </summary>
		public DateTime CustomDaylightSavingsStart
		{
			get
			{
				return _customDaylightSavingsStart;
			}
			set
			{
				if( _customDaylightSavingsStart == value )
					return;					

				_customDaylightSavingsStart = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4803"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4804"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// End date for daylight savings time for a given year.
		/// </summary>
		public DateTime CustomDaylightSavingsEnd
		{
			get
			{
				return _customDaylightSavingsEnd;
			}
			set
			{
				if( _customDaylightSavingsEnd == value )	
					return;

				_customDaylightSavingsEnd = value;
				OnPropertyChanged();
			}
		}
	}
}
